/*********************************************************************
 *
 *  HyperText Transfer Protocol (HTTP) Server
 *  Module for Microchip TCP/IP Stack
 *   -Serves dynamic pages to web browsers such as Microsoft Internet 
 *    Explorer, Mozilla Firefox, etc.
 *	 -Reference: RFC 2068
 *
 **********************************************************************
 * FileName:        HTTP.c
 * Dependencies:    TCP, MPFS, HTTPGetVar() callback, HTTPExecCmd() callback
 * Processor:       PIC18, PIC24F, PIC24H, dsPIC30F, dsPIC33F, PIC32
 * Compiler:        Microchip C32 v1.05 or higher
 *					Microchip C30 v3.12 or higher
 *					Microchip C18 v3.30 or higher
 *					HI-TECH PICC-18 PRO 9.63PL2 or higher
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * Copyright (C) 2002-2009 Microchip Technology Inc.  All rights
 * reserved.
 *
 * Microchip licenses to you the right to use, modify, copy, and
 * distribute:
 * (i)  the Software when embedded on a Microchip microcontroller or
 *      digital signal controller product ("Device") which is
 *      integrated into Licensee's product; or
 * (ii) ONLY the Software driver source files ENC28J60.c, ENC28J60.h,
 *		ENCX24J600.c and ENCX24J600.h ported to a non-Microchip device
 *		used in conjunction with a Microchip ethernet controller for
 *		the sole purpose of interfacing with the ethernet controller.
 *
 * You should refer to the license agreement accompanying this
 * Software for additional information regarding your rights and
 * obligations.
 *
 * THE SOFTWARE AND DOCUMENTATION ARE PROVIDED "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT
 * LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * MICROCHIP BE LIABLE FOR ANY INCIDENTAL, SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF
 * PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR SERVICES, ANY CLAIMS
 * BY THIRD PARTIES (INCLUDING BUT NOT LIMITED TO ANY DEFENSE
 * THEREOF), ANY CLAIMS FOR INDEMNITY OR CONTRIBUTION, OR OTHER
 * SIMILAR COSTS, WHETHER ASSERTED ON THE BASIS OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE), BREACH OF WARRANTY, OR OTHERWISE.
 *
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Nilesh Rajbharti     8/14/01     Original
 * Nilesh Rajbharti     9/12/01     Released (Rev. 1.0)
 * Nilesh Rajbharti     2/9/02      Cleanup
 * Nilesh Rajbharti     5/22/02     Rev 2.0 (See version.log for detail)
 * Nilesh Rajbharti     7/9/02      Rev 2.1 (Fixed HTTPParse bug)
 * Howard Schlunder		2/9/05		Fixed variable substitution 
 *									parsing (uses hex now)
 ********************************************************************/
/*

Modified from the original Microchip file by Mauro Grassi to add SD card file system support

*/

#define __HTTP_C

#include "h/maindemo.h"
#include "h/TCPIPConfig.h"
#include "h/common.h"
#include "h/GenericTypeDefs.h"

#if defined(STACK_USE_HTTP_SERVER)

#include "h/TCPIP.h"
//#include "ctype.h"
#include "h/common.h"

// HTTP File Types
//#define HTTP_TXT        (0u)
//#define HTTP_HTML       (1u)
//#define HTTP_CGI        (2u)
//#define HTTP_XML        (3u)
//#define HTTP_GIF        (4u)
//#define HTTP_PNG        (5u)
//#define HTTP_JPG        (6u)
//#define HTTP_JAVA       (7u)
//#define HTTP_WAV        (8u)
//#define HTTP_PDF        (9u)
//#define HTTP_DAT        (10u)
//#define HTTP_CGH        (11u)

// Each entry in this structure must be in UPPER case.
// Order of these entries must match with those defined by "HTTP File Types" defines.
static ROM FILE_TYPES ROMhttpFiles[] =
{
    { "HTM" },          // HTTP_HTML 
    { "TXT" },          // HTTP_TXT
    { "CGI" },          // HTTP_CGI
    { "XML" },          // HTTP_XML
    { "GIF" },          // HTTP_GIF
    { "PNG" },          // HTTP_PNG
    { "JPG" },          // HTTP_JPG
    { "CLA" },          // HTTP_JAVA
    { "WAV" },          // HTTP_WAV
    { "PDF" },          // PDF
    { "DAT" },          // DAT
    { "CGH" },          // CGH another dynamic type but not executable
	{ ""    }			// HTTP_UNKNOWN
};

BYTE httpFiles[TOTAL_FILE_TYPES][SHORT_STRING_MAX];

static ROM BYTE ROMhttpFilesPermissions[]
=
{
    // bit definitions are
    // bit 0 = whether file requires password
    // bit 1 = whether file is dynamic
    // bit 2 = whether file is executable
    2,
    1,
    7,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    1,
    3,
    0
};

long httpFilesPermissions[TOTAL_FILE_TYPES];

typedef struct _HTTP_CONTENT
{
    ROM BYTE* typeString;
} HTTP_CONTENT;

// Content entry order must match with those "HTTP File Types" define's.
static ROM HTTP_CONTENT ROMhttpContents[] =
{
    { (ROM BYTE*)"text/html" },                    // HTTP_HTML
    { (ROM BYTE*)"text/plain" },                   // HTTP_TXT
    { (ROM BYTE*)"text/html" },                    // HTTP_CGI
    { (ROM BYTE*)"text/xml" },                     // HTTP_XML
    { (ROM BYTE*)"image/gif" },                    // HTTP_GIF
    { (ROM BYTE*)"image/png" },                    // HTTP_PNG
    { (ROM BYTE*)"image/jpeg" },                   // HTTP_JPG
    { (ROM BYTE*)"application/java-vm" },          // HTTP_JAVA
    { (ROM BYTE*)"audio/x-wave" },                 // HTTP_WAV
    { (ROM BYTE*)"application/pdf"},               // HTTP_PDF
    { (ROM BYTE*)"application/octet-stream"},	   // DAT
    { (ROM BYTE*)"text/html" },                    // HTTP_CGH
    { (ROM BYTE*)"application/octet-stream"}       // HTTP_UNKNOWN
};

BYTE httpContents[TOTAL_FILE_TYPES][SHORT_STRING_MAX];

#define TOTAL_HTTP_CONTENTS     (13)

// HTTP FSM states for each connection.
typedef enum _SM_HTTP
{
    SM_HTTP_IDLE = 0u,
    SM_HTTP_GET,
    SM_HTTP_NOT_FOUND,
    SM_HTTP_GET_READ,
    SM_HTTP_GET_PASS,
    SM_HTTP_GET_DLE,
    SM_HTTP_GET_HANDLE,
    SM_HTTP_GET_HANDLE_NEXT,
    SM_HTTP_GET_VAR,
    SM_HTTP_HEADER,
    SM_HTTP_DISCARD
} SM_HTTP;

// Supported HTTP Commands
typedef enum _HTTP_COMMAND
{
    HTTP_GET = 0u,
    HTTP_POST,
    HTTP_NOT_SUPPORTED,
    HTTP_INVALID_COMMAND
} HTTP_COMMAND;

// HTTP Connection Info - one for each connection.
typedef struct _HTTP_INFO
{
    TCP_SOCKET socket;
    FIL file;
    SM_HTTP smHTTP;
    BYTE smHTTPGet;
    WORD VarRef;
    BYTE bProcess;
    BYTE Variable;
    BYTE fileType;
} HTTP_INFO;
typedef BYTE HTTP_HANDLE;

typedef enum
{
    HTTP_NOT_FOUND = 0u,
    HTTP_NOT_AVAILABLE
} HTTP_MESSAGES;

// Following message order must match with that of HTTP_MESSAGES enum.
static ROM BYTE *HTTPMessages[] =
{
	    (ROM BYTE*)"HTTP/1.1 404 File Not Found\r\n\r\nFile Not Found.\r\n",
	    (ROM BYTE*)"HTTP/1.1 503 \r\n\r\nService Unavailable\r\n"
};

// Standard HTTP messages.
static ROM BYTE HTTP_OK_STRING[] = "HTTP/1.1 200 OK\r\nContent-type: ";
static ROM BYTE HTTP_OK_NO_CACHE_STRING[] = "HTTP/1.1 200 OK\r\nExpires: Wed, 05 Apr 2006 02:00:05 GMT\r\nCache-control: private\r\nContent-type: ";
static ROM BYTE HTTP_LOGIN_STRING[]= "HTTP/1.1 401 Unauthorized\r\nWWW-Authenticate: Basic realm=\"Login Required\"\r\nConnection: close\r\nContent-type: text/html\r\n\r\n401 Unauthorized: Password required\r\n";

#define HTTP_OK_STRING_LEN (sizeof(HTTP_OK_STRING)-1)
#define HTTP_LOGIN_STRING_LEN (sizeof(HTTP_LOGIN_STRING)-1)
#define HTTP_OK_NO_CACHE_STRING_LEN (sizeof(HTTP_OK_NO_CACHE_STRING)-1)

static ROM BYTE HTTP_HEADER_END_STRING[] = "\r\n\r\n";
#define HTTP_HEADER_END_STRING_LEN (sizeof(HTTP_HEADER_END_STRING)-1)

// HTTP Command Strings
static ROM BYTE HTTP_GET_STRING[] = "GET";
#define HTTP_GET_STRING_LEN (sizeof(HTTP_GET_STRING)-1)

// Default HTML file.
static ROM BYTE HTTP_DEFAULT_FILE_STRING[] = "INDEX.HTM";
#define HTTP_DEFAULT_FILE_STRING_LEN (sizeof(HTTP_DEFAULT_FILE_STRING)-1)

// Maximum nuber of arguments supported by this HTTP Server.
#define MAX_HTTP_ARGS       (11u)

// Maximum HTML Command String length.
#define MAX_HTML_CMD_LEN    (100u)


static HTTP_INFO HCB[MAX_HTTP_CONNECTIONS];


static void HTTPProcess(HTTP_HANDLE h);
static HTTP_COMMAND HTTPParse(BYTE *string,
                              BYTE** arg,
                              BYTE* argc,
                              BYTE* type);

static BOOL SendFile(HTTP_INFO* ph);
BYTE* strcpyram2ram(BYTE*, BYTE*);
BOOL FTPVerify(BYTE*, BYTE*);
long ftpPort;
long ftpDataPort;
long smtpPort;
long httpPort;
char lastPage[STRING_MAX];				// holds the last valid page.
BYTE lastFileType;
WORD Base64Decode(BYTE*, WORD, BYTE*, WORD);

void initFileTypes(void)
{
    int i;
    
    for(i=0; i<TOTAL_FILE_TYPES; i++)
    {
    strcpypgm2ram((char*)httpFiles[i], (char*)ROMhttpFiles[i].fileExt);
    httpFilesPermissions[i]=(long)ROMhttpFilesPermissions[i];
    strcpypgm2ram((char*)httpContents[i], (char*)ROMhttpContents[i].typeString);
    }
}

/*********************************************************************
 * Function:        void HTTPInit(void)
 *
 * PreCondition:    TCP must already be initialized.
 *
 * Input:           None
 *
 * Output:          HTTP FSM and connections are initialized
 *
 * Side Effects:    None
 *
 * Overview:        Set all HTTP connections to Listening state.
 *                  Initialize FSM for each connection.
 *
 * Note:            This function is called only one during lifetime
 *                  of the application.
 ********************************************************************/
void HTTPInit(void)
{
    BYTE i;
    for ( i = 0; i <  MAX_HTTP_CONNECTIONS; i++ )
    {
        HCB[i].socket = TCPOpen(0, TCP_OPEN_SERVER, httpPort, TCP_PURPOSE_HTTP_SERVER);
        HCB[i].smHTTP = SM_HTTP_IDLE;
    }
    
}

/*********************************************************************
 * Function:        void HTTPServer(void)
 *
 * PreCondition:    HTTPInit() must already be called.
 *
 * Input:           None
 *
 * Output:          Opened HTTP connections are served.
 *
 * Side Effects:    None
 *
 * Overview:        Browse through each connections and let it
 *                  handle its connection.
 *                  If a connection is not finished, do not process
 *                  next connections.  This must be done, all
 *                  connections use some static variables that are
 *                  common.
 *
 * Note:            This function acts as a task (similar to one in
 *                  RTOS).  This function performs its task in
 *                  co-operative manner.  Main application must call
 *                  this function repeatdly to ensure all open
 *                  or new connections are served on time.
 ********************************************************************/
void HTTPServer(void)
{
    BYTE conn;

    for ( conn = 0;  conn < MAX_HTTP_CONNECTIONS; conn++ )
        HTTPProcess(conn);
}


/*********************************************************************
 * Function:        static BOOL HTTPProcess(HTTP_HANDLE h)
 *
 * PreCondition:    HTTPInit() called.
 *
 * Input:           h   -   Index to the handle which needs to be
 *                          processed.
 *
 * Output:          Connection referred by 'h' is served.
 *
 * Side Effects:    None
 *
 * Overview:
 *
 * Note:            None.
 ********************************************************************/
BYTE httpData[MAX_HTML_CMD_LEN+1];	// for storing the GET command line in the HTTP header.
BYTE unlocked;						// =0 if the password/username does not match the set values =1 if the password/username match the set values
BYTE hasPassword;					// =0 if there is no password field in the HTTP header =1 if there is a password field in the HTTP header
BYTE reqPassword;                   // =0 if no password required =1 if a password is required...
BYTE passData[MAX_HTML_CMD_LEN+1];	// username/password information
BYTE epassData[MAX_HTML_CMD_LEN+1];

int HTTPUnlocked(void)
{
    // returns 1 if and only if there is a password, it is correct or a password is not required...
    if((reqPassword==0)||((hasPassword==1)&&(unlocked==1)))return 1;
    return 0;
}

static void HTTPProcess(HTTP_HANDLE h)
{
    BYTE* strptr;
	BYTE lineFeed[3];
	static BYTE restData[MAX_HTML_CMD_LEN+1];			// making this static solves the problem where file permissions set to private weren't requesting password on certain IP addresses. (version 5.40 fix) M.G.
    HTTP_COMMAND httpCommand;
    BOOL lbContinue;
    BYTE *arg[MAX_HTTP_ARGS];
    BYTE argc;
    HTTP_INFO* ph;
	WORD w;
	FRESULT res;
	WORD p, q;

    ph = &HCB[h];
    #if(DEBUG_STACK_HTTP)
	putrsUART("#5.1");
    #endif
    do
    {
        lbContinue = FALSE;
        // If during handling of HTTP socket, it gets disconnected,
        // forget about previous processing and return to idle state.
        if(!TCPIsConnected(ph->socket))
        {
            reqPassword=1;
            unlocked=0; 
			hasPassword=0;
			ph->smHTTP = SM_HTTP_IDLE;
            break;
        }
        #if(DEBUG_STACK_HTTP)
	    putrsUART("#5.2");
        #endif
        switch(ph->smHTTP)
        {
        case SM_HTTP_IDLE:
            #if(DEBUG_STACK_HTTP)
            putrsUART("#5.3");
            #endif
			// Search for the CRLF deliminating the end of the first GET/HEAD/POST request
			w = TCPFindROMArray(ph->socket, (ROM BYTE*)"\r\n", 2, 0, FALSE);
            if(w == 0xFFFFu)
			{
				if(TCPGetRxFIFOFree(ph->socket) == 0)
				{
					// Request is too big, we can't support it.
					TCPDisconnect(ph->socket);
				}
				break;
			}
            lbContinue = TRUE;
			p= TCPFindROMArray(ph->socket, (ROM BYTE*)"GET", 3, 0, FALSE);
			q= TCPFindROMArray(ph->socket, (ROM BYTE*)"Basic ", 6, 0, FALSE);
			if(w > sizeof(restData)-1)
				w = sizeof(restData)-1;
			TCPGetArray(ph->socket, restData, w);
            restData[w] = '\0';
			TCPGetArray(ph->socket, lineFeed, 2);
            #if(DEBUG_STACK_HTTP)
			putrsUART("Proc Line: ");
			putrsUART(restData);
			putrsUART("\r\n");
            #endif
			if(p==0)
			{ 
				// found GET
				strcpyram2ram(httpData, restData); 
				httpCommand = HTTPParse(restData, arg, &argc, &ph->fileType);
				if((httpFilesPermissions[ph->fileType] & REQ_PASSWORD)==0)reqPassword=0;            
			}

			if(q<w)
			{
				// Found Authentication
				hasPassword=1;
                strptr=strcpyram2ram((BYTE*)epassData, (BYTE*)userName);
                *strptr++=':';
                strptr=strcpyram2ram((BYTE*)strptr, (BYTE*)passWord);
                Base64Encode(epassData, strptr-epassData, passData, sizeof(passData));
                if(!strcmppgm2ram((char*)passData, (char*)&restData[q+6]))unlocked=1; else unlocked=0;
			}

			if(restData[0]!='\0')break;

            //#if(DEBUG_STACK_HTTP)
            //#endif
		    ph->smHTTP = SM_HTTP_NOT_FOUND;
            argc = MAX_HTTP_ARGS;
            httpCommand = HTTPParse(httpData, arg, &argc, &ph->fileType);
            if(httpCommand==HTTP_GET)
            {
                // If there are any arguments, this must be a remote command.
                // Execute it and then send the file.
                // The file name may be modified by command handler.
				if(argc>1u)
                {
                    // all executable commands require a password
                    reqPassword=1;
                    HTTPExecCmd(&arg[0], argc);
                    // Command handler must have modified arg[0] which now
                    // points to actual file that will be sent as a result of
                    // this remote command.
                    // Assume that Web author will only use CGI or HTML
                    // file for remote command.
                    ph->fileType = lastFileType;
                    strcpyram2ram((BYTE*)arg[0], (BYTE*)lastPage);    
                } 
                
                res=f_open(&ph->file, (const char*)arg[0], FA_READ | FA_OPEN_EXISTING);
                if((res != FR_OK)||(!HTTPUnlocked()))
                {
					// here if invalid M.G.
                    ph->Variable = HTTP_NOT_FOUND;
                    ph->smHTTP = SM_HTTP_NOT_FOUND;
                }
                else
                {
                    // store the last validly served static page...
                    if((httpFilesPermissions[ph->fileType] & IS_EXECUTABLE)!=0)
                    {
                                strcpyram2ram((BYTE*)lastPage, (BYTE*)arg[0]);
                                lastFileType = ph->fileType;         
                    }
                    ph->smHTTP = SM_HTTP_HEADER;
                }
            }
            break;

        case SM_HTTP_NOT_FOUND:
#if(DEBUG_STACK_HTTP)
	putrsUART("#5.4");
#endif
            if(TCPIsPutReady(ph->socket))
            {
				if(!HTTPUnlocked())
				{
				TCPPutROMArray(ph->socket, (ROM BYTE*)HTTP_LOGIN_STRING, HTTP_LOGIN_STRING_LEN);
                //TCPPutROMString(ph->socket, httpContents[ph->fileType].typeString);
				TCPPutROMArray(ph->socket, (ROM BYTE*)HTTP_HEADER_END_STRING, HTTP_HEADER_END_STRING_LEN);
				} else
				{
				TCPPutROMString(ph->socket, HTTPMessages[ph->Variable]);
				}
				TCPFlush(ph->socket);
				TCPDisconnect(ph->socket);
				ph->smHTTP = SM_HTTP_IDLE;
            }
            break;

        case SM_HTTP_HEADER:
                #if(DEBUG_STACK_HTTP)
	            putrsUART("#5.5");
                #endif
                if ( TCPIsPutReady(ph->socket) )
                {
                lbContinue = TRUE;
                if((httpFilesPermissions[ph->fileType] & IS_DYNAMIC)!=0)
				{
                    ph->bProcess = TRUE;
					TCPPutROMArray(ph->socket, (ROM BYTE*)HTTP_OK_NO_CACHE_STRING, HTTP_OK_NO_CACHE_STRING_LEN);
				}
                else
				{
                    ph->bProcess = FALSE;
                    TCPPutROMArray(ph->socket, (ROM BYTE*)HTTP_OK_STRING, HTTP_OK_STRING_LEN);
				}
                TCPPutROMString(ph->socket, (BYTE*)httpContents[ph->fileType]);
				TCPPutROMArray(ph->socket, (ROM BYTE*)HTTP_HEADER_END_STRING, HTTP_HEADER_END_STRING_LEN);
                ph->smHTTPGet = SM_HTTP_GET_READ;
                ph->smHTTP = SM_HTTP_GET;
                }
                break;

        case SM_HTTP_GET:
			    // Throw away any more data receieved - we aren't going to use it.
                #if(DEBUG_STACK_HTTP)
	                putrsUART("#5.6");
                #endif

			    TCPDiscard(ph->socket);

                if(SendFile(ph))
                {
                //MPFSClose();
				f_close(&ph->file);
				TCPDisconnect(ph->socket);
                ph->smHTTP = SM_HTTP_IDLE;
                }
                #if(DEBUG_STACK_HTTP)
	                putrsUART("#5.61");
                #endif
                break;

		default:
                #if(DEBUG_STACK_HTTP)
	                putrsUART("#5.7");
                #endif
        		break;
        }
    } while( lbContinue );
    
    #if(DEBUG_STACK_HTTP)
		putrsUART("#5.8");
    #endif
}


/*********************************************************************
 * Function:        static BOOL SendFile(HTTP_INFO* ph)
 *
 * PreCondition:    None
 *
 * Input:           ph      -   A HTTP connection info.
 *
 * Output:          File reference by this connection is served.
 *
 * Side Effects:    None
 *
 * Overview:        None
 *
 * Note:            None.
 ********************************************************************/
static BOOL SendFile(HTTP_INFO* ph)
{
    BOOL lbTransmit=0;
    BYTE c;
	WORD w;
    WORD_VAL HexNumber;
	UINT n;
	FRESULT res;

	// Check if file is dynamic (.cgi) -- need to look for and 
	// process escape sequences
	if(ph->bProcess)					
	{
		w = TCPIsPutReady(ph->socket);
		while(w)
		{
	        lbTransmit = FALSE;
	
	        if(ph->smHTTPGet != SM_HTTP_GET_VAR)
	        {
				res=f_read(&ph->file, &c, 1, &n);
				if((res!=FR_OK)||(n<1))
				{
					f_close(&ph->file);
					TCPFlush(ph->socket);
					return TRUE;
				}
	        }
	
			switch(ph->smHTTPGet)
			{
			case SM_HTTP_GET_READ:
			    if ( c == HTTP_VAR_ESC_CHAR )
			        ph->smHTTPGet = SM_HTTP_GET_DLE;
			    else
			        lbTransmit = TRUE;
			    break;
			
			case SM_HTTP_GET_DLE:
			    if ( c == HTTP_VAR_ESC_CHAR )
			    {
			        lbTransmit = TRUE;
			        ph->smHTTPGet = SM_HTTP_GET_READ;
			    }
			    else
			    {
			        HexNumber.v[1] = c;
			        ph->smHTTPGet = SM_HTTP_GET_HANDLE;
			    }
			    break;
			
			case SM_HTTP_GET_HANDLE:
			    HexNumber.v[0] = c;
			    ph->Variable = hexatob(HexNumber);
			
			    ph->smHTTPGet = SM_HTTP_GET_VAR;
			    ph->VarRef = HTTP_START_OF_VAR;
			    break;
			
			case SM_HTTP_GET_VAR:
			    ph->VarRef = HTTPGetVar(ph->Variable, ph->VarRef, &c);
			    lbTransmit = TRUE;
			    if ( ph->VarRef == HTTP_END_OF_VAR )
			        ph->smHTTPGet = SM_HTTP_GET_READ;
			    break;
			}
			
			if(lbTransmit)
			{
			    TCPPut(ph->socket, c);
				w--;
			}
		}
    }	
	else	// Static page content -- no processing required
	{
		w = TCPIsPutReady(ph->socket);
		while(w >= sizeof(fileBuffer))
		{
				res=f_read(&ph->file, fileBuffer, sizeof(fileBuffer), &n);
				if((res!=FR_OK)||(n<sizeof(fileBuffer)))
				{
					f_close(&ph->file);
					TCPPutArray(ph->socket, fileBuffer, n);
					TCPFlush(ph->socket);
					return TRUE;
				}
			    TCPPutArray(ph->socket, fileBuffer, sizeof(fileBuffer));
			    w -= sizeof(fileBuffer);
			    lbTransmit = TRUE;
		}
		if(lbTransmit)
			TCPFlush(ph->socket);
	}
    return FALSE;
}

/*********************************************************************
 * Function:        static HTTP_COMMAND HTTPParse(BYTE *string,
 *                                              BYTE** arg,
 *                                              BYTE* argc,
 *                                              BYTE* type)
 *
 * PreCondition:    None
 *
 * Input:           string      - HTTP Command String
 *                  arg         - List of string pointer to hold
 *                                HTTP arguments.
 *                  argc        - Pointer to hold total number of
 *                                arguments in this command string/
 *                  type        - Pointer to hold type of file
 *                                received.
 *                              Valid values are:
 *                                  HTTP_TXT
 *                                  HTTP_HTML
 *                                  HTTP_GIF
 *                                  HTTP_CGI
 *                                  HTTP_UNKNOWN
 *
 * Output:          HTTP FSM and connections are initialized
 *
 * Side Effects:    None
 *
 * Overview:        None
 *
 * Note:            This function parses URL that may or may not
 *                  contain arguments.
 *                  e.g. "GET HTTP/1.0 thank.htm?name=MCHP&age=12"
 *                      would be returned as below:
 *                          arg[0] => GET
 *                          arg[1] => thank.htm
 *                          arg[2] => name
 *                          arg[3] => MCHP
 *                          arg[4] => 12
 *                          argc = 5
 *
 *                  This parses does not "de-escape" URL string.
 ********************************************************************/
static HTTP_COMMAND HTTPParse(BYTE *string,
                            BYTE** arg,
                            BYTE* argc,
                            BYTE* type)
{
    BYTE i;
    BYTE smParse;
    HTTP_COMMAND cmd;
    BYTE *ext;
    BYTE c;
//    ROM BYTE *fileType;

    enum
    {
        SM_PARSE_IDLE,
        SM_PARSE_ARG,
        SM_PARSE_ARG_FORMAT
    };

    smParse = SM_PARSE_IDLE;
    ext = NULL;
    i = 0;

    // Only "GET" is supported for time being.
    if ( !memcmppgm2ram(string, (ROM void*) HTTP_GET_STRING, HTTP_GET_STRING_LEN) )
    {
        string += (HTTP_GET_STRING_LEN + 1);
        cmd = HTTP_GET;
    }
    else
    {
        return HTTP_NOT_SUPPORTED;
    }

    // Skip white spaces.
    while( *string == ' ' )
        string++;

    c = *string;

    while ( c != ' ' &&  c != '\0' && c != '\r' && c != '\n' )
    {
        // Do not accept any more arguments than we haved designed to.
        if ( i >= *argc )
            break;

        switch(smParse)
        {
        case SM_PARSE_IDLE:
            arg[i] = string;
            c = *string;
            if ( c == '/' || c == '\\' )
                smParse = SM_PARSE_ARG;
            break;

        case SM_PARSE_ARG:
            arg[i++] = string;
            smParse = SM_PARSE_ARG_FORMAT;
            /*
             * Do not break.
             * Parameter may be empty.
             */

        case SM_PARSE_ARG_FORMAT:
            c = *string;
            if ( c == '?' || c == '&' )
            {
                *string = '\0';
                smParse = SM_PARSE_ARG;
            }
            else
            {
                // Recover space characters.
                if ( c == '+' )
                    *string = ' ';

                // Remember where file extension starts.
                else if ( c == '.' && i == 1u )
                {
                    ext = ++string;
                }

                else if ( c == '=' )
                {
                    *string = '\0';
                    smParse = SM_PARSE_ARG;
                }

                // Only interested in file name - not a path.
                //else if ( c == '/' || c == '\\' )
                //    arg[i-1] = string+1;

            }
            break;
        }
        string++;
        c = *string;
    }
    *string = '\0';

    *type = HTTP_UNKNOWN;
    if (ext != NULL)
    {
        ext = (BYTE*)strupr((char*)ext);
        for ( c = 0; c < (TOTAL_FILE_TYPES-1); c++ )
        {
            if (!stricmppgm2ram((BYTE*)ext, (BYTE*)httpFiles[c]))
            {
                *type = c;
                break;
            }
            //fileType += sizeof(FILE_TYPES);
        }
    }

    if ( i == 0u )
    {
        memcpypgm2ram(arg[0], (ROM void*)HTTP_DEFAULT_FILE_STRING,
                                     HTTP_DEFAULT_FILE_STRING_LEN);
        arg[0][HTTP_DEFAULT_FILE_STRING_LEN] = '\0';
        *type = 0;
        i++;
    }
    *argc = i;

    return cmd;
}

#endif //#if defined(STACK_USE_HTTP_SERVER)
